#!/usr/bin/env python3

#   Gimp-Python - allows the writing of Gimp plugins in Python.
#   Copyright (C) 1997  James Henstridge <james@daa.com.au>
#
#   This program is free software: you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 3 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program.  If not, see <https://www.gnu.org/licenses/>.

# ToDo: añadir syntaxhighlight

import gettext, os, sys
from gtkutils import *
import traceback
import pyconsole2
import warnings
from gi.repository import GLib, GObject, Gtk, Gdk, GdkPixbuf
from gi.repository import GimpUi
from gi.repository import Gimp
from iconsdialog import *
from snippetnew import *

import gi
gi.require_version('Gimp', '3.0')
gi.require_version('GimpUi', '3.0')
gi.require_version('Gtk', '3.0')

from gimpshelf import *

thisPth = os.path.dirname(sys.argv[0]) + os.sep
shelf=GimpShelf(thisPth+"snippets" + os.sep + "snippets.json")

sys.stderr = open(thisPth + os.sep + "errors.txt",'a')
sys.stdout=sys.stderr

jSon = [
    {"t":"Common","l":[
    {"t":"Get 1st image","c":"img1=Gimp.get_images()[0]"}
]},
    {"t":"Image","l":[
    {"t":"Get width","c":"w=image.get_width()"},
    {"t":"Get height","c":"w=image.get_height()"}
]},
    {"t":"Gtk","l":[
    {"t":"All imports","c":"from gi.repository import Gtk\nimport gi\ngi.require_version('Gtk', '3.0')\ngi.require_version('Gdk', '3.0')\ngi.require_version('Gimp', '3.0')"}
]}
]

txDebug = Gtk.TextView(buffer=None)

def detError(e):
    Gimp.message(f"Error: {e}")
    Gimp.message(traceback.format_exc())

def selfDbg(e):
    try:
        tb = txDebug.get_buffer()
        p = tb.get_end_iter()
        tb.insert(p, str(e)+"\n")
        tb.insert(p, traceback.format_exc() or "\n")
        tb.insert(p, "_ _ _ _ _ _\n")
    except Exception:
        detError(Exception)

shelfSnippets=shelf.read_all()
if shelfSnippets=={} or shelfSnippets==None:
    shelf.write(jSon)
    shelfSnippets=shelf.read_all()

DOMAIN = "python-console-2"
LOCALE_DIR = os.path.join(os.path.dirname(__file__), 'locale')
gettext.bindtextdomain(DOMAIN, LOCALE_DIR)
gettext.textdomain(DOMAIN)
_ = gettext.gettext

PROC_NAME = 'python-fu-console-2'

RESPONSE_BROWSE, RESPONSE_CLEAR, RESPONSE_SAVE, SAVE_CLEAN, ICONS_BROWSER = range(5)

def run(procedure, config, data):
    GimpUi.init("python-console-2.py")
    namespace = {'__builtins__': __builtins__,
                 '__name__': '__main__', '__doc__': None,
                 'Babl': gi.repository.Babl,
                 'cairo': gi.repository.cairo,
                 'Gdk': gi.repository.Gdk,
                 'Gegl': gi.repository.Gegl,
                 'Gimp': gi.repository.Gimp,
                 'GimpUi': gi.repository.GimpUi,
                 'Gio': gi.repository.Gio,
                 'Gtk': gi.repository.Gtk,
                 'GdkPixbuf': gi.repository.GdkPixbuf,
                 'GLib': gi.repository.GLib,
                 'GObject': gi.repository.GObject,
                 'Pango': gi.repository.Pango}
    
    css = b"""
    treeview:selected {background-color: #8080ff; color: black;}
    """

    provider = Gtk.CssProvider()
    provider.load_from_data(css)
    Gtk.StyleContext.add_provider_for_screen(
        Gdk.Screen.get_default(),
        provider,
        Gtk.STYLE_PROVIDER_PRIORITY_APPLICATION
    )

    class GimpConsole(pyconsole2.Console):
        def __init__(self, quit_func=None, initial_history=None):
            banner = ('GIMP %s Python Console\nPython %s\n' %
                      (Gimp.version(), sys.version))
            pyconsole2.Console.__init__(self,
                                        locals=namespace, banner=banner,
                                        quit_func=quit_func,
                                        initial_history=initial_history)

        def _commit(self):
            pyconsole2.Console._commit(self)
            Gimp.displays_flush()

    class ConsoleDialog(GimpUi.Dialog):

        def __init__(self):
            GRT = Gtk.ResponseType
            use_header_bar = Gtk.Settings.get_default().get_property("gtk-dialogs-use-header")
            GimpUi.Dialog.__init__(self, use_header_bar=use_header_bar)
            self.set_property("help-id", PROC_NAME)
            Gtk.Window.set_title(self, _("Python Console"))
            Gtk.Window.set_role(self, PROC_NAME)
            dlgBtns(self, {_("_Save"): GRT.OK, _("Cl_ear"): RESPONSE_CLEAR, _("Save clean"): SAVE_CLEAN, 
                           _("Icons browser"): ICONS_BROWSER,
                           _("_Browse..."): RESPONSE_BROWSE,  _("_Close"): GRT.CLOSE})
            Gtk.Widget.set_name(self, PROC_NAME)
            GimpUi.Dialog.set_alternative_button_order_from_array(self, [GRT.CLOSE,
                                                                         RESPONSE_BROWSE,
                                                                         RESPONSE_CLEAR,
                                                                         SAVE_CLEAN,
                                                                         ICONS_BROWSER,
                                                                         GRT.OK])
            self.set_keep_above(True)
            history = config.get_property('history')
            self.cons = GimpConsole(quit_func=lambda: Gtk.main_quit(), initial_history=history)
            self.style_set(None, None)
            self.connect('response', self.response)
            self.connect('style-set', self.style_set)
            self.browse_dlg = None
            self.save_dlg = None
            self.set_default_size(900, 600)
            self.set_border_width(5)
            self.connect("destroy", Gtk.main_quit)

            hpaned = Gtk.HPaned()
            hpaned.set_wide_handle(True)
            contents_area = Gtk.Dialog.get_content_area(self)
            contents_area.pack_start(hpaned, True, True, 0)
            hpaned.show()
            hpaned.set_position(500)

            scrl_win = scrollW(self.cons)
            hpaned.pack1(scrl_win, True, True)
            self.tabs = []
            self.nB = Gtk_(Gtk.Notebook(), {})
            self.nB.set_size_request(150, -1)
            self.tabTits = [_("Inspect"), _("Snippets"), _("Debug")]
            for n in range(0, len(self.tabTits)):
                t = Gtk_(Gtk.VBox(), {"margin": 5})
                self.tabs.append(t)
                tabTit = GtkLb(self.tabTits[n], {'margin_bottom': 5, 'margin_left': 3, 'margin_right': 3, 'margin_top': 3})
                self.nB.append_page(t, tabTit)
            hpaned.pack2(self.nB)
            self.tabs[0].add(self.nb1())
            self.tabs[1].add(self.nbSnp())
            self.tabs[2].add(self.nbDbg())
            self.docBuff = ''
            self.show_all()
            self.inspectclick(self.objProps)

########################################
        # revisar
        def snp_cntxt_menu(self, event):
            menu = Gtk.Menu()
            model = self.tvSn.get_model()
            delete_item = Gtk.MenuItem(label=_("Delete")+ " "+'"' + model[self.tvSnp_sel][0] + '"')
            menu.append(delete_item)
            menu.show_all()
            menu.popup_at_pointer(event)

        def nbSnp(self):
            self.pannedSnp=None
            vb = Gtk.VBox()
            hb = Gtk_(Gtk.HBox(),p=vb, fill=0)
            GtkBtn(_('Create new'), {}, p=hb, cnn={"clicked": self.snpNewW}, f=0)
            self.tsSnippets = Gtk.TreeStore(str,str, GdkPixbuf.Pixbuf)
            self.tvSn = GtkTreeV(self.tsSnippets, [_('Snippets')], rowAct = self.snpAct, btnPress = self.delSnp)
            self.tvSn.colPixbuf(2, tit="")
            self.tvSn.set_size_request(300, 240)
            self.snpTvFill()
            vb.add(scrollW(self.tvSn))
            vb.show()
            self.snpEdit = GTKCbLab(_("Edit before insert"),props={"active":True}, toggled= self.viewCode, p=vb)
            self.snpAutoExec = GTKCbLab(_("Auto execute"),p=vb)
            self.codevb=Gtk.VBox()
            self.precode=scrollW(p=self.codevb)
            self.precodeTx=Gtk_(Gtk.TextView(),{}, p = self.precode)
            GtkBtn(_('Insert'), {}, p=self.codevb, cnn={"clicked": self.add2code}, f=0)
            self.pannedSnp = Gtk.VPaned()
            self.pannedSnp.set_wide_handle(True)
            self.pannedSnp.pack1(vb, True, True)
            self.pannedSnp.pack2(self.codevb, True, True)
            return self.pannedSnp

        def delSnp(self, tree_view, event):
            if event.type == Gdk.EventType.BUTTON_PRESS and event.button == 1:
                path_info = tree_view.get_path_at_pos(int(event.x), int(event.y))
                if path_info is not None:
                    path, column, cell_x, cell_y = path_info
                    if column == tree_view.get_column(1):
                        tree_iter = self.tsSnippets.get_iter(path)
                        if tree_iter:
                            tit = self.tsSnippets.get_value(tree_iter, 0)
                            thisParent = self.tsSnippets.iter_parent(tree_iter)
                            if thisParent != None:
                                parent = self.tsSnippets.get_value(thisParent, 0)
                                snippet_del(shelf, parent, tit)
                                self.tsSnippets.remove(tree_iter)
                                return True
                            else:
                                if not self.tsSnippets.iter_has_child(tree_iter):
                                    snippet_del_grp(shelf, tit)
                                    self.tsSnippets.remove(tree_iter)
                                    return True
            return False

        def add2code(self, widget):
            fn =  TextViewGetAll(self.precodeTx)
            if self.snpAutoExec.get_active():
                fn+= '\n\n'
            self.add2buffer(fn)

        def viewCode(self, widget):
            if self.pannedSnp !=None:
                h = self.pannedSnp.get_allocated_height()
                self.pannedSnp.set_position(h/2 if widget.get_active() else h)

        def snpTvFill(self):
            ts=self.tvSn.get_model()
            ts.clear()
            pb = get_icon_pixbuf("edit-delete-symbolic")
            for n in shelf.read_all():
                tit, lst = (n["t"],n["l"])
                iter = ts.append(None,[tit,'', pb if len(lst)==0 else None])
                for i in lst:
                    ts.append(iter,[i['t'],i['c'], pb])

        def snpGrps(self):
            grps=[(k['t'],i) for i, (k) in enumerate(shelf.read_all())]
            grps.append([_("+ New group"),-1])
            return grps

        def snpNewW(self, widget):
            dialog = SnippetNew(parent=self, title=_("Add Snippet"))
            text, newGrp = dialog.run_and_get_text()
            if text is not None:
                self.snpTvFill()
                if newGrp:
                    fillCombo(self.snpGrps(), self.cmbGroup)

        def nbDbg(self):
            global txDebug
            txDebug.set_wrap_mode(Gtk.WrapMode.WORD)
            scrl2 = scrollW(txDebug)
            return scrl2

        def openIconsViewer(self):
            dialog = IconsViewer(parent=self, title=_("Icons viewer"))
            text = dialog.run_and_get_text()
            if text is not None:
                self.add2buffer(text)

        def nb1(self):
            vpaned = Gtk.VPaned()
            vpaned.set_wide_handle(True)
            vb = Gtk_(Gtk.VBox())
            secPos = Gtk.EntryIconPosition.SECONDARY
            #self.entry = Gtk_(Gtk.Entry(), fill=0,cnn={'activate': self.inspectclick})
            self.entry = GtkTx("Gimp", f=0,cnn={'activate': self.inspectclick})
            #self.entry.set_text("Gimp")
            self.entry.set_placeholder_text(_("Variable to inspect..."))
            self.entry.set_icon_from_icon_name(secPos, "system-search-symbolic")
            self.entry.set_icon_activatable(secPos, True)
            self.entry.connect("icon-press", self.inspectclick2, secPos)
            self.toInspect = lbEntry(vb, _('Inspect:'), self.entry)
            self.lsProps = Gtk.ListStore(str, str)
            self.propsFilter = self.lsProps.filter_new()
            cols = [_('Inspecting Gimp'), _('Type')]
            self.objProps = GtkTreeV(self.lsProps, cols=cols, rowAct = self.actRow)
            self.tree_sel = self.objProps.get_selection()
            self._tree_sel_selchg_id = self.tree_sel.connect("changed", self.onSelChanged)
            vb.add(scrollW(self.objProps))
            self.txFilter = Gtk_(Gtk.SearchEntry(), p=vb, fill=0, cnn={"changed": self.filterTree}, tt=_("Show only te values containing this text"))
            self.txFilter.props.placeholder_text = _('Filter')            
            self.chMagic = GTKCbLab(_("Hide magic methods"), toggled= self._chck_magic, p=vb)
            vb.show()
            vpaned.pack1(vb, True, True)
            vpaned.show()
            self.propsFilter.set_visible_func(self.visible_cb, None)
            self.objProps.set_model(self.propsFilter)
            vb2 = Gtk.VBox()
            scrl3 = scrollW(vb2)
            self.docLab = GtkLb("", {"use_markup": True, "selectable":True, 
                                     "halign":Gtk.Align.START, "valign":Gtk.Align.START}, p=vb2)
            self.docLab.set_line_wrap(True)
            self.enumOpts = Gtk_(Gtk.VBox(), None, vb2)

            vpaned.pack2(scrl3, True, False)
            return vpaned

        def _chck_magic(self, check_btn):
            self.propsFilter.refilter()

        def visible_cb(self, model, iter, data):
            tx = self.txFilter.get_text().strip()
            # si está marcado magic
            este = str(model.get_value(iter, 0))
            magic=True
            filter=True
            if self.chMagic.get_active()==True:
                magic= este.find("__")==-1
            if tx != '':
                filter = este.lower().find(tx.lower()) > -1
            return filter and magic

        # mio
        def inspectclick(self, widget):
            try:
                self.tree_sel.disconnect(self._tree_sel_selchg_id)
                self.inspectObj(self.toInspect.get_text())
                self._tree_sel_selchg_id = self.tree_sel.connect("changed", self.onSelChanged)
            except Exception:
                selfDbg(Exception)

        def inspectclick2(self, widget,a,b,c):
            try:
                self.tree_sel.disconnect(self._tree_sel_selchg_id)
                self.inspectObj(self.toInspect.get_text())
                self._tree_sel_selchg_id = self.tree_sel.connect(
                    "changed", self.onSelChanged)
            except Exception:
                selfDbg(Exception)

        def inspectObj(self, tx):
            self.lsProps.clear()
            self.cons.do_command(tx)
            tmp = eval(tx, self.cons.locals)
            try:
                if hasattr(tmp, '__doc__'):
                    if tmp.__doc__ != None:
                        self.docBuff = tmp.__doc__ + ''
                self.objProps.get_columns()[0].set_title(_('Inspecting ') + tx)
                all = dir(tmp)
                for ii in all:
                    try:
                        if hasattr(tmp, ii):
                            self.lsProps.append([ii, ''])
                    except Exception:
                        self.lsProps.append(["? " + ii, 'ERROR'])
                        selfDbg(Exception)
                for n in self.lsProps:
                    v = self.lsProps.get_value(n.iter, 0)
                    if v.find("?") == -1:
                        tmp = eval(tx + '.' + v, self.cons.locals)
                        self.lsProps.set_value(n.iter, 1, type(tmp).__name__)
            except Exception:
                selfDbg(Exception)

        def filterTree(self, extra=None):
            self.propsFilter.refilter()

        def onSelChanged(self, tree_selection):
            vals = {'v': '', 'doc': ''}
            tx = self.toInspect.get_text()
            self.cons.do_command(tx)
            (model, iter) = tree_selection.get_selected()
            if iter == None:
                return
            value, tipo = (model.get_value(iter, 0), model.get_value(iter, 1))
            txObj = tx+'.'+value
            tmp = None
            enums = self.enumOpts.get_children()
            for n in enums:
                self.enumOpts.remove(n)
            if value.find("?") != -1:
                return
            docstr = ""
            try:
                if value+'' != 'clean_all':
                    tmp = eval(txObj, self.cons.locals)
                if tipo in ['str', 'tuple', 'int', 'float']:
                    vals['v'] = tmp
                elif tipo == 'PDBFunction':
                    proc_name = str(tmp.proc_name)
                    docstr += Tag("b,big", proc_name) + "\n"
                    docstr += Tag("i", pdbProcType[int(tmp.proc_type)]) + "\n"
                    if tmp.proc_help != '':
                        docstr += str(tmp.proc_help) + "\n"
                    docstr += Tag("b", _("Parameters"))+"\n"
                    for n in range(tmp.nparams):
                        nParam = tmp.params[n]
                        docstr += nParam[1]+"\n" + nParam[2]+"\n"
                    if len(tmp.return_vals) != 0:
                        docstr += Tag("b", _("Return values"))+"\n"
                        for n in tmp.return_vals:
                            docstr += Tag("b", GimpPDBArgType[int(n[0])]) + "\n" + str(n[1])+"\n" + str(n[2])+"\n"
                    if tmp.proc_author != '':
                        docstr += Tag("b", _("Author:"))+" " + \
                            str(tmp.proc_author)+"\n"
                    docstr += str(tmp.proc_copyright)+"\n" + \
                        str(tmp.proc_date)+"\n"
                    return
                if hasattr(tmp, '__doc__'):
                    if tmp.__doc__ != None:
                        vals['doc'] = tmp.__doc__
                docstr += Tag("b", Tag("big", txObj))+"\n"
                if ((vals['v'] == '') and (vals['doc'] == '')):
                    docstr += Tag("i", tipo) + ""
                    if tipo.casefold().find("enum") > 0:
                        try:
                            lll = list(tmp)
                            for n in lll:
                                GtkBtn(str(n.name),{}, cnn={"clicked": self.pressEnum},f=0, p=self.enumOpts)
                        except Exception:
                            txDebug(Exception)
                else:
                    docstr += tipo
                    if (vals['v'] != ''):
                        docstr += "=<span color='#00ff00'>" + str(vals['v'])+"</span>\n"
                    if (vals['doc'] != ''):
                        docstr += "\n" + str(vals['doc']) + "\n"
                self.docLab.set_markup(docstr)
            except Exception:
                detError(Exception)

        #######################################################################
        def add2buffer(self, tx):
            buffer = self.cons.buffer
            buffer.insert_at_cursor(tx)
            self.cons.grab_focus()

        def pressEnum(self, a):
            seltx = ""
            model, tree_iter = getSel(self.objProps)
            if tree_iter:
                seltx = model.get_value(tree_iter, 0)
            self.add2buffer(self.toInspect.get_text() + "." + seltx + "." + a.get_label())

        def snpAct(self, treeview, path, column):
            model = treeview.get_model()
            iter = model.get_iter(path)
            fn = model.get_value(iter, 1)
            if column.get_title()!="":

                if self.snpAutoExec.get_active():
                    fn+= '\n\n'
                if self.snpEdit.get_active():
                    bufferAdd(self.precodeTx, fn)
                else:
                    self.add2buffer(fn)
                return True
            else:
                return False

        def actRow(self, treeview, path, column):
            model = treeview.get_model()
            iter = model.get_iter(path)
            tx = self.toInspect.get_text()
            fn = model.get_value(iter, 0)
            if model.get_value(iter, 1) == 'PDBFunction':
                tmp = eval(tx+'.'+fn, self.cons.locals)
                params = ''
                for n in range(tmp.nparams):
                    if params != '':
                        params = params+' ,'
                    params = params + str(tmp.params[n][1])
                if params != '':
                    fn = fn+'('+params+')'
                retVals = ''
                if len(tmp.return_vals) != 0:
                    for n in tmp.return_vals:
                        if retVals != '':
                            retVals = retVals+', '
                        retVals = retVals + str(n[1])
                if retVals != '':
                    tx = retVals+' = '+tx
            self.add2buffer(tx+'.'+fn)
            return
        # fin mio

        def style_set(self, old_style, user_data):
            pass

        def response(self, dialog, response_id):
            if response_id == RESPONSE_BROWSE:
                self.browse()
            elif response_id == SAVE_CLEAN:
                self.saveClean = 1
                self.save_dialog()
            elif response_id == ICONS_BROWSER:
                self.openIconsViewer()
            elif response_id == RESPONSE_CLEAR:
                self.cons.banner = None
                self.cons.clear()
            elif response_id == Gtk.ResponseType.OK:
                self.saveClean = 0
                self.save_dialog()
            else:
                # Store up to 100 commands.
                config.set_property('history', self.cons.history.items[-100:])
                Gtk.main_quit()
            self.cons.grab_focus()

        def command_for_procedure(self, proc_name):
            '''
            Assemble string of Python code that when eval'd will call proc_name
            with contrived arguments.

            The purpose is to generate a template for a call to the PDB procedure
            the user has selected.
            The call MIGHT work as is.
            Otherwise, the names of the arguments might be enough that the user
            can figure out how to edit the template.

            The code will run in the environment of the console/browser, which
            is not the GIMP v2 GimpFu environment but the GIMP v3 PyGObject
            introspected environment.

            If ever GimpFu module is resurrected, and Python console imports it,
            then revert this code to its v2 form.
            '''
            proc = Gimp.get_pdb().lookup_procedure(proc_name)
            if proc is None:
                return None

            return_values = proc.get_return_values()
            param_specs = proc.get_arguments()

            cmd = f"procedure = Gimp.get_pdb().lookup_procedure('{proc_name}'); "
            cmd += f"config = procedure.create_config(); "
            for arg in param_specs:
                argName = arg.name
                if argName == 'run-mode':
                    # Special handling for run mode.
                    cmd += f"config.set_property('{argName}', Gimp.RunMode.INTERACTIVE); "
                else:
                    cmd += "config.set_property('" + argName + "', " + argName.replace('-', '_') + "); "
            cmd += f"result = procedure.run(config); "
            cmd += f"success = result.index(0)"
            if len(return_values) > 0:
                i = 1
                for retval in return_values:
                    cmd += '; {} = result.index({})'.format(retval.name.replace('-', '_'), i)
                    i += 1
            return cmd

        def browse_response(self, dlg, response_id):
            if response_id != Gtk.ResponseType.APPLY:
                Gtk.Widget.hide(dlg)
                return

            proc_name = dlg.get_selected()
            if not proc_name:
                return

            cmd = self.command_for_procedure(proc_name)
            if cmd is None:
                return

            buffer = self.cons.buffer
            lines = buffer.get_line_count()
            iter = buffer.get_iter_at_line_offset(lines - 1, 4)
            buffer.delete(iter, buffer.get_end_iter())
            buffer.place_cursor(buffer.get_end_iter())
            buffer.insert_at_cursor(cmd)
            # not insert a newline, user can edit and then "enter" the command

        def browse(self):
            if not self.browse_dlg:
                use_header_bar = Gtk.Settings.get_default().get_property("gtk-dialogs-use-header")
                dlg = GimpUi.ProcBrowserDialog(use_header_bar=use_header_bar)
                Gtk.Window.set_title(dlg, _("Python Procedure Browser"))
                Gtk.Window.set_role(dlg, PROC_NAME)
                GRT = Gtk.ResponseType
                dlgBtns(dlg, {_("_Apply"): GRT.APPLY, _("_Close"): GRT.CLOSE})
                Gtk.Dialog.set_default_response(self, GRT.OK)
                GimpUi.Dialog.set_alternative_button_order_from_array(
                    dlg, [GRT.CLOSE, GRT.APPLY])
                dlg.connect('response', self.browse_response)
                dlg.connect('row-activated', lambda dlg: dlg.response(GRT.APPLY))
                self.browse_dlg = dlg
            showOnTop(self.browse_dlg)

        def save_response(self, dlg, response_id):
            GRT = Gtk.ResponseType
            if response_id == GRT.DELETE_EVENT:
                self.save_dlg = None
                return
            elif response_id == GRT.OK:
                fname = dlg.get_filename()
                try:
                    logfile = open(fname, 'w')
                except IOError as e:
                    Gimp.message(_("Could not open '%s' for writing: %s") % (
                        fname, e.strerror))
                    return
                log = bufferAll(self.cons)
                try:
                    if self.saveClean == 1:
                        log = self.cleanCode(log)
                    logfile.write(log)
                    logfile.close()
                except IOError as e:
                    Gimp.message(
                        _("Could not write to '%s': %s") % (fname, e.strerror))
                    return
            Gtk.Widget.hide(dlg)

        def cleanCode(self, ll):
            lines = ll.split("\n")
            limpio = []
            for l in lines:
                if (l.find("...") == 0 or l.find(">>>") == 0):
                    limpio.append(l.replace("... ", "").replace(">>> ", ""))
            return "\n".join(limpio)

        def save_dialog(self):
            if not self.save_dlg:
                GRT = Gtk.ResponseType
                _fcd = Gtk.FileChooserDialog
                dlg = _fcd(action=Gtk.FileChooserAction.SAVE)
                Gtk.Window.set_title(dlg, _("Save Python-Fu Console Output"))
                Gtk.Window.set_transient_for(dlg, self)
                dlgBtns(dlg, {_("_Cancel"): GRT.CANCEL, _("_Save"): GRT.OK})
                Gtk.Dialog.set_default_response(self, GRT.OK)
                with warnings.catch_warnings():
                    warnings.filterwarnings('ignore')
                    _fcd.set_alternative_button_order_from_array(
                        dlg, [GRT.OK, GRT.CANCEL])
                dlg.connect('response', self.save_response)
                self.save_dlg = dlg
            showOnTop(self.save_dlg)

        def run(self):
            Gtk.Widget.show_all(self)
            Gtk.main()

    ConsoleDialog().run()
    return procedure.new_return_values(Gimp.PDBStatusType.SUCCESS, GLib.Error())

class PythonConsole (Gimp.PlugIn):
    ## GimpPlugIn virtual methods ##
    def do_set_i18n(self, name):
        return True, 'gimp30-console2', None

    def do_query_procedures(self):
        return [PROC_NAME]

    def do_create_procedure(self, name):
        if name == PROC_NAME:
            proc = Gimp.Procedure.new(self, name,
                                      Gimp.PDBProcType.PLUGIN,
                                      run, None)
            proc.set_menu_label(_("Python _Console Extended"))
            proc.set_documentation(_("Interactive GIMP Python interpreter"),
                                   _("Type in commands and see results"),
                                   "")
            proc.set_attribution("James Henstridge",
                                 "James Henstridge",
                                 "1997-1999")
            proc.add_enum_argument("run-mode", _("Run mode"),
                                   _("The run mode"), Gimp.RunMode,
                                   Gimp.RunMode.INTERACTIVE,
                                   GObject.ParamFlags.READWRITE)
            proc.add_string_array_aux_argument("history", "Command history",
                                               "Command history",
                                               GObject.ParamFlags.READWRITE)
            proc.add_menu_path("<Image>/Filters/Development")
            return proc
        return None

Gimp.main(PythonConsole.__gtype__, sys.argv)
# 686 685 701 809