import numpy as np
from mathutils import Vector as V

class GeoSphere:
    def __init__(self):
        self.vertices = np.array([[1, 0, 0],[0, 1, 0],[0, 0, 1]])
        self.faces = np.array([[0, 1, 2]])
        self.latx=[1,2]
        self.laty=[0,2]
        self.latz=[0,1]

    def normalize(self, v):
        """Normaliza un vector para que esté en la superficie de una esfera de radio 1"""
        return v / np.linalg.norm(v)
    
    def subdivide_geodesic(self):
        """Subdivide cada cara del poliedro actual, colocando los nuevos vértices sobre la esfera."""
        new_faces = []
        midpoint_cache = {}
        vertices = self.vertices.tolist()

        def midpoint(v1, v2):
            """ punto medio entre dos vértices y lo normaliza sobre la esfera"""
            key = tuple(sorted((v1, v2)))
            if key not in midpoint_cache:
                # Calcular y almacenar el punto medio normalizado
                midpoint_cache[key] = len(vertices)
                mid = self.normalize((np.array(vertices[v1]) + np.array(vertices[v2])) / 2)
                vertices.append(mid.tolist())
                if (mid[0]==0): self.latx.insert(v1,len(vertices)-1)
                if (mid[1]==0): self.laty.insert(v1,len(vertices)-1)
                if (mid[2]==0): self.latz.insert(v1,len(vertices)-1)
            return midpoint_cache[key]

        for face in self.faces:
            # Obtener índices de los vértices de cada cara
            v1, v2, v3 = face
            # Calcular puntos medios de cada lado de la cara
            a = midpoint(v1, v2)
            b = midpoint(v2, v3)
            c = midpoint(v3, v1)
            # Crear cuatro nuevas caras usando los puntos medios
            new_faces.extend(([v1, a, c],[a, v2, b],[c, b, v3],[a, b, c]))
        # Actualizar los vértices y las caras con las subdivisiones realizadas
        self.vertices = np.array(vertices)
        self.faces = np.array(new_faces)
    
    def create_geodesic_sphere(self, subdivisions=2):
        """Genera una esfera geodésica subdividiendo el octaedro base varias veces."""
        for _ in range(subdivisions): self.subdivide_geodesic()

    def transform(self, translate, scale, moveIndex=0, flip=0):
        vcopy = np.copy(self.vertices)

        listIx=[{"i":n,"v":vcopy[n]} for n in self.latx]
        listIx.sort(key=lambda x:x["v"][1])
        ix=[x["i"] for x in listIx]

        listIy=[{"i":n,"v":vcopy[n]} for n in self.laty]
        listIy.sort(key=lambda x:x["v"][2])
        iy =[x["i"] for x in listIy]  

        listIz=[{"i":n,"v":vcopy[n]} for n in self.latz]
        listIz.sort(key=lambda i:i["v"][0])
        iz=[x["i"] for x in listIz]

        v = [n*scale+translate for n in vcopy]
        f = np.copy(self.faces)
        if moveIndex>0 or flip>0:
            for n in range(len(ix)): ix[n] = ix[n] + moveIndex
            for n in range(len(iy)): iy[n] = iy[n] + moveIndex
            for n in range(len(iz)): iz[n] = iz[n] + moveIndex
            for n in range(len(f)):
                if moveIndex>0:
                    f[n] = f[n] + (moveIndex,moveIndex,moveIndex)
                if flip>0:
                    f[n]=np.flip(f[n])
        return v,f,ix,iy,iz