import bpy
from bpy.types import Scene
from bpy.props import FloatProperty, EnumProperty
from bpy.types import Operator


bl_info = {
    "name": "Merge auto",
    "author": "Paco Garcia",
    "version": (1, 0, 0),
    "blender": (4, 5, 0),
    "location": "3D View > Edit Mode > M key",
    "description": "Overwrites M shortcut",
    "category": "Interface",
}

orig_kmap_items = []


class MESH_OT_merge_auto(Operator):
    bl_idname = "mesh.merge_auto"
    bl_label = "Mensaje Personalizado"
    bl_description = "Muestra un mensaje personalizado en lugar de merge"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH' and context.active_object is not None

    def execute(self, context):
        opt = bpy.context.scene.merge_type
        if opt != 'DISTANCE':
            bpy.ops.mesh.merge(type=opt)
        else:
            dist = bpy.context.scene.merge_distance
            bpy.ops.mesh.remove_doubles(threshold=dist)
        return {'FINISHED'}


class MESH_OT_restore_merge(Operator):
    bl_idname = "mesh.restore_orig_merge"
    bl_label = "Restore original merge "
    bl_description = "Restores the M shortcut to his original function (merge)"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        restore_orig_keymap()
        return {'FINISHED'}


Scene.merge_type = EnumProperty(
    items=[
        ("CENTER", "At center", "Merge selected at center"),
        ("CURSOR", "At cursor", "Merge selected at cursor"),
        ("COLLAPSE", "Collapse", "Collapse selected points"),
        ("FIRST", "At first", "Merge selected at first"),
        ("LAST", "At last", "Merge selected at last"),
        ("DISTANCE", "By distance", "Merge selected by distance")
    ], name="Merge type", description="Select the mode for merge.")
Scene.merge_distance = FloatProperty(
    name="Merge distance", default=0.0001, precision=4)


class VIEW3D_PT_merge_plus_panel(bpy.types.Panel):
    bl_label = "Merge auto"
    bl_idname = "VIEW3D_PT_merge_override"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Tool'

    def draw(self, context):
        layout = self.layout

        box = layout.box()
        box.label(text="Merge:")
        col = box.column(align=True)
        col.prop(context.scene, "merge_type", expand=True)
        col = box.column(align=True)
        col.prop(context.scene, "merge_distance", expand=True)


def override_merge_keymap():
    global orig_kmap_items

    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        km = kc.keymaps.get('Mesh')
        if not km:
            km = kc.keymaps.new(name='Mesh', space_type='EMPTY')
        for kmi in km.keymap_items:
            if kmi.type == 'M' and kmi.idname == 'mesh.merge':
                orig_kmap_items.append({'idname': kmi.idname, 'type': kmi.type, 'value': kmi.value,
                                        'shift': kmi.shift, 'ctrl': kmi.ctrl, 'alt': kmi.alt,
                                        'oskey': kmi.oskey, 'key_modifier': kmi.key_modifier,
                                        'active': kmi.active})
                kmi.active = False
        kmi = km.keymap_items.new('mesh.merge_auto', type='M', value='PRESS')
        return True
    return False


def restore_orig_keymap():
    global orig_kmap_items

    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        km = kc.keymaps.get('Mesh')
        if km:
            for kmi in km.keymap_items:
                if kmi.idname == 'mesh.merge_auto':
                    km.keymap_items.remove(kmi)
            for kmi in km.keymap_items:
                if kmi.type == 'M' and kmi.idname == 'mesh.merge':
                    kmi.active = True
            orig_kmap_items.clear()
            return True
    return False


def remove_custom_keymap():
    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        km = kc.keymaps.get('Mesh')
        if km:
            for kmi in list(km.keymap_items):
                if kmi.idname == 'mesh.merge_auto':
                    km.keymap_items.remove(kmi)


classes = (MESH_OT_merge_auto, MESH_OT_restore_merge,
           VIEW3D_PT_merge_plus_panel)


def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    override_merge_keymap()


def unregister():
    restore_orig_keymap()
    remove_custom_keymap()
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)


if __name__ == "__main__":
    register()
