import bpy
from bpy.types import Operator, PropertyGroup
from bpy.props import FloatVectorProperty, IntProperty, PointerProperty, FloatProperty, EnumProperty, BoolProperty
from bpy_extras.object_utils import AddObjectHelper, object_data_add
from mathutils import Vector as V, Matrix
import math
from . tools import *


def on_enum_change(self, context):
    objs = self.objs
    if (objs != 'None'):
        vals = objBBox(objs)
        self.cntr1 = vals[0]
        self.cntr2 = vals[1]
    return None


def icoId(name):
    return pcoll[name].icon_id


def addConeCyl2_button(self, context):
    global pcoll
    self.layout.operator(OBJECT_OT_addCyl2.bl_idname,
                         text="Cone/cylinder from 2 centers", icon_value=icoId('MESH_CYL'))


class cyl2props(PropertyGroup):
    cntr1desc = "First center for the object"
    cntr2desc = "Second center for the object"

    typeItems = (('rect', "Rect", "Rect", icoId('CONE_EDGE_RECT'), 0),
                 ('bevel', "Bevel", "Bevel all the edges",
                  icoId('CONE_EDGE_BEVEL'), 1),
                 ('round', "Rounded", "Rounded", icoId('CONE_EDGE_ROUND'), 2),
                 ('stud', "Stud", "Stud", icoId('CONE_EDGE_STUD'), 3),
                 ('scoop', "Scoop", "Scoop", icoId('CONE_EDGE_SCOOP'), 4))

    cntr1: FloatVectorProperty(name="Center 1", description=cntr1desc, default=(
        0, 0, 0), subtype='TRANSLATION')  # type: ignore
    r1: FloatProperty(name="Radius 1", description=cntr1desc,
                      min=0.0, default=2.5)  # type: ignore
    lnkRads: BoolProperty(name="linked Rads", default=False)  # type: ignore
    rEff1: FloatProperty(name="Size", description=cntr1desc,
                         min=0.0, default=0.5)  # type: ignore
    type1: EnumProperty(
        name="Edge", description="Choose the type of edge", items=typeItems)  # type: ignore
    eff1def: IntProperty(name="Effect divisions", description="Divisions",
                         default=3, min=3, max=200)  # type: ignore
    cntr2: FloatVectorProperty(name="Center 2", description=cntr2desc, default=(
        0, 2, 0), subtype='TRANSLATION')  # type: ignore
    r2: FloatProperty(name="Radius 2", description=cntr1desc,
                      min=0.0, default=1)  # type: ignore
    rEff2: FloatProperty(name="Size", description=cntr1desc,
                         min=0.0,  default=1.0)  # type: ignore
    type2: EnumProperty(
        name="Edge", description="Choose the type of edge", items=typeItems)  # type: ignore
    eff2def: IntProperty(name="Effect divisions", description="Divisions",
                         default=3, min=3, max=200)  # type: ignore
    vertices: IntProperty(name="Vertices", description="Vertices",
                          default=12, min=3, max=150)  # type: ignore
    objs: EnumProperty(name="Object", items=list_objs,
                       update=on_enum_change)  # type: ignore
    spheres: BoolProperty(name="Connected spheres", description="Creates two connected spheres form de centers and radius",
                          default=False)  # type: ignore


def addCyl2(self, context):
    sCt = self.cntrs
    V1, V2 = (sCt.cntr1, sCt.cntr2)
    r1 = sCt.r1
    r2 = r1 if sCt.lnkRads == True else sCt.r2
    rEff1, rEff2, typeA, typeB = (sCt.rEff1, sCt.rEff2, sCt.type1, sCt.type2)
    vertices = sCt.vertices
    def1, def2 = (sCt.eff1def, sCt.eff2def)
    if sCt.spheres:
        vrts, edges, faces = cone2pntsSph(V1, r1, V2, r2, vertices, def1, def2)
    else:
        vrts, edges, faces = cone2pnts(
            V1, r1, V2, r2, rEff1, rEff2, typeA, typeB, vertices, def1, def2)
    meshname = 'cone2'
    mesh = bpy.data.meshes.new(name=meshname)
    self.meshname = mesh.name
    mesh.from_pydata(vrts, edges, faces)
    mesh.shade_smooth()
    mesh.update(calc_edges=True)
    mesh.validate(verbose=False)
    obj = object_data_add(context, mesh, operator=self)
    bpy.ops.object.shade_auto_smooth(angle=math.radians(40))
    rotateTo(obj, V1, V2)


def rotateTo(obj, v1, v2):
    v1 = V(v1)
    v2 = V(v2)
    direction = (v2 - v1).normalized()
    rot_matrix = direction.to_track_quat('Y', 'Z').to_matrix().to_4x4()
    loc_matrix = Matrix.Translation(v1)
    obj.matrix_world = loc_matrix @ rot_matrix
    return obj


def vectH(lay, var, varName, label=""):
    if label != "":
        lay.label(text=label)
    row = lay.row(align=True)
    row.prop(var, varName, index=0, text="x")
    row.prop(var, varName, index=1, text="y")
    row.prop(var, varName, index=2, text="z")
    return row


class OBJECT_OT_addCyl2(Operator, AddObjectHelper):
    """Create a new Mesh Object"""
    import os
    bl_idname = 'mesh.add_cyl_2'
    bl_label = "Cylinder/cone from 2 centers"
    bl_icon = os.path.join(os.path.dirname(__file__), 'icons', 'mesh_cyl')
    bl_options = {'REGISTER', 'UNDO'}
    cntrs: PointerProperty(
        type=cyl2props, name='cntrs', description="Centers of the cylinder/cone")  # type: ignore

    @classmethod
    def poll(cls, context):
        return True

    def propsType(self, type, num, row, lay):
        icon = 'LOCKED' if self.cntrs.lnkRads else 'UNLOCKED'
        row.prop(self.cntrs, "lnkRads", text="", icon=icon, toggle=True)
        if self.cntrs.spheres:
            row.prop(self.cntrs, 'eff'+num+'def')
        else:
            row.prop(self.cntrs, 'type'+num)
            if type != "rect":
                row = lay.row()
                row.prop(self.cntrs, 'rEff'+num)
                if type not in ["bevel", "stud"]:
                    row.prop(self.cntrs, 'eff'+num+'def')

    def draw(self, context):
        lay = self.layout
        sCt = self.cntrs
        vectH(lay, sCt, "cntr1", label="Center 1:")
        col = lay.column()
        row = col.row()
        row.prop(sCt, 'r1')
        self.propsType(sCt.type1, '1', row, lay)
        vectH(lay, sCt, "cntr2", label="Center 2:")
        row = lay.row()
        row.prop(sCt, 'r1' if sCt.lnkRads == True else 'r2')
        self.propsType(sCt.type2, '2', row, lay)
        lay.prop(sCt, 'vertices')
        lay.prop(sCt, 'spheres')
        lay.label(text='Centers from object bounding box corners')
        lay.prop(sCt, 'objs')

    def execute(self, context):
        self.location = (0.0, 0.0, 0.0)
        addCyl2(self, context)
        return {'FINISHED'}


def addCyl2_button(self, context):
    global pcoll
    self.layout.operator(OBJECT_OT_addCyl2.bl_idname,
                         text="Cylinder/cone from centers", icon_value=pcoll['MESH_TOR'].icon_id)


def ang2pRad(A, B, C):
    AB = B-A
    AC = C-A
    lnABC = AB.length * AC.length or 1
    return math.acos(AB.dot(AC)/lnABC)


def ang2p(A, B, C):
    return math.degrees(ang2pRad(A, B, C))


def coneDef(B, arr):
    arr.append(B)


def coneRnd(A, B, C, R, arr, res):
    if R == 0:
        arr.append(B)
        return
    uBA = (A - B).normalized()
    uBC = (C - B).normalized()
    Angle = math.acos(uBA.dot(uBC))
    Dist = R / math.sin(Angle / 2)
    Center = B + Dist * (uBA + uBC).normalized()
    tDist = math.sqrt(Dist*Dist - R*R)
    ang1 = ang2p(Center, B + uBA * tDist, B + uBC * tDist)

    incRot = ang1/res
    rr = 0 if (B.y == 0) else 180-ang1

    for n in range(res+1):
        p = rotV(V((0, -R, 0)), rr+incRot*n, "Z", Center)
        arr.append(p)
    return


def rotV(vv, rot, axis='Z', translate=None):
    matriz_rotacion_z = Matrix.Rotation(math.radians(rot), 3, axis)
    rv = matriz_rotacion_z @ vv
    if translate:
        rv += translate
    return rv


def coneScoop(A, B, C, R, arr, res):
    if R == 0:
        arr.append(B)
        return
    ang1 = ang2p(B, A, C)
    for n in range(res+1):
        rot = ang1/res*n
        if B.y == 0:
            p = rotV(V((-R, 0, 0)), -rot, 'Z', B)
        else:
            p = rotV(V((R, 0, 0)), -180+ang1-rot, 'Z', B)
        arr.append(p)


def coneStud(A, B, C, R, arr):
    if R == 0:
        arr.append(B)
        return
    hh = math.cos(ang2pRad(B, A, C))*R
    hh2 = 0 if hh > 0 else hh

    if (B.y == 0):
        arr += [B - V((R+hh2, 0, 0)), B -
                V((R+hh2, -R, 0)), B + V((-hh, R, 0))]
    else:
        arr += [B+V((-hh, -R, 0)), B+V((-R-hh2, -R, 0)), B+V((-R-hh2, 0, 0))]


def coneBevel(A, B, C, R, arr):
    if R == 0:
        arr.append(B)
        return
    ang1 = ang2p(B, A, C)
    if B.y == 0:
        arr += [B+V((-R, 0, 0)), rotV(V((-R, 0, 0)), -ang1, 'Z', B)]
    else:
        arr += [rotV(V((R, 0, 0)), -180+ang1, 'Z', B), B - V((R, 0, 0))]


def joinSpheres(r1, r2, ln, res1, res2, arr):
    if r1 == 0:
        return
    rot = math.degrees(math.atan2(r2-r1, ln))
    pRot = (90-rot)/res1
    for n in range(res1+1):
        arr.append(rotV(V((0, -r1, 0)), pRot*n, 'Z'))
    pRot = (rot+90)/res2
    for n in range(res2+1):
        arr.append(rotV(V((0, -r2, 0)), 90-rot+pRot*n, 'Z', V((0, ln, 0))))


def polis3(nums, faces, center, offs):
    for i in range(nums-1):
        faces.append([center, offs + i+1, offs + i])
    faces.append([center, offs, offs + nums-1])


def polis4(nums, faces, offs):
    for i in range(nums-1):
        faces.append([offs + i, offs + i+1, offs +
                     i + 1 + nums, offs + i + nums])
    faces.append([offs, offs+nums, offs + nums*2-1,  offs + nums-1])


def cone2pntsSph(V1, r1, V2, r2, numVerts, def1, def2):
    vrts = []
    edges = []
    faces = []
    arrPrf = []
    a, b = (V1, V2)
    ln = (b-a).length
    joinSpheres(r1, r2, ln, def1, def2, arrPrf)
    lArr = len(arrPrf)
    vrts.append(V((0, -r1, 0)))
    incRot = 360/numVerts
    for n in range(1, lArr-1):
        for i in range(numVerts):
            vrts.append(rotV(arrPrf[n], incRot*i, 'Y'))
    vrts.append(V((0, ln + r2, 0)))
    polis3(numVerts, faces, 0, 1)
    for i in range(lArr):
        polis4(numVerts, faces, numVerts * i+1)
    offs = len(vrts) - numVerts - 2
    i1 = len(vrts)-1
    for n in range(1, numVerts):
        faces.append([i1, offs + n, offs + n+1])
    faces.append([i1, i1-1, i1 - numVerts])
    return vrts, edges, faces


def cone2pnts(V1, r1, V2, r2, ef1, ef2, type1, type2, vertices, def1, def2):
    vrts = []
    edges = []
    faces = []
    arrPrf = []
    a, b = (V1, V2)
    ln = (b-a).length

    vr = [V((0, 0, 0)),  V((r1, 0, 0)), V((r2, ln, 0)), V((r2-ef2, ln, 0))]
    match type1:
        case "rect":     coneDef(vr[1], arrPrf)
        case "round":    coneRnd(V((r1-ef1, 0, 0)), vr[1], vr[2], ef1, arrPrf, def1)
        case "scoop":  coneScoop(vr[0], vr[1], vr[2], ef1, arrPrf, def1)
        case "bevel":  coneBevel(vr[0], vr[1], vr[2], ef1, arrPrf)
        case "stud":    coneStud(vr[0], vr[1], vr[2], ef1, arrPrf)
    match type2:
        case "rect":     coneDef(vr[2], arrPrf)
        case "round":    coneRnd(vr[1], vr[2], vr[3], ef2, arrPrf, def2)
        case "scoop":  coneScoop(vr[1], vr[2], vr[3], ef2, arrPrf, def2)
        case "bevel":  coneBevel(vr[1], vr[2], vr[3], ef2, arrPrf)
        case "stud":    coneStud(vr[1], vr[2], vr[3], ef2, arrPrf)

    pp = vertices
    lArr = len(arrPrf)
    faceA = []
    faceB = []
    for i in range(pp):
        rot = 360/pp*i
        for n in range(lArr):
            vv = arrPrf[n]
            vrts.append(rotV(vv, rot, 'Y'))
            if n == 0:
                faceA.append(i*lArr)
            if n == lArr-1:
                faceB.append((i+1)*lArr-1)
            if i > 0 and n < lArr-1:
                faces.append([i*lArr+n, i*lArr+n+1, (i-1)
                             * lArr+n+1, (i-1) * lArr+n])
    faceA.reverse()
    # bpy.ops.mesh.spin(steps=32, angle=2*pi, center=(0,0,0), axis=(0,0,1))

    for n in range(lArr):
        if n > 0 and n < lArr-1:
            faces.append([n, n+1, (pp-1) * lArr+n+1, (pp-1) * lArr+n])
    faces.append([0, 1, lArr*(pp-1)+1, lArr*(pp-1)])

    if len(faceA) > 0:
        faces.append(faceA)

    if len(faceB) > 0:
        faces.append(faceB)

    return vrts, edges, faces
# 410