import bpy
from bpy.types import Scene
from bpy.props import FloatProperty, EnumProperty
from bpy.types import Operator
import blf

bl_info = {
    "name": "Merge auto",
    "author": "Paco Garcia",
    "version": (1, 0, 0),
    "blender": (4, 5, 0),
    "location": "3D View > Edit Mode > M key",
    "description": "Overwrites M shortcut",
    "category": "Interface",
}

orig_kmap_items = []


class MESH_OT_merge_auto(Operator):
    bl_idname = "mesh.merge_auto"
    bl_label = "Merge"
    bl_description = "Overwrites M shortcut"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH' and context.active_object is not None

    def execute(self, context):
        opt = bpy.context.scene.merge_type
        if opt != 'DISTANCE':
            bpy.ops.mesh.merge(type=opt)
        else:
            dist = bpy.context.scene.merge_distance
            bpy.ops.mesh.remove_doubles(threshold=dist)
        return {'FINISHED'}


class MESH_OT_del_auto(Operator):
    bl_idname = "mesh.del_auto"
    bl_label = "Delete"
    bl_description = "Overwrites DEL shortcut"
    bl_options = {'REGISTER', 'UNDO'}

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH' and context.active_object is not None

    def execute(self, context):
        del_type = context.scene.del_type
        bom = bpy.ops.mesh
        if del_type in ['VERT', 'EDGE',  'FACE', 'EDGE_FACE', 'ONLY_FACE']:
            bom.delete(type=context.scene.del_type)
        else:
            self.report({'INFO'}, del_type)
            if del_type == 'DISSOLVE_V':
                bom.dissolve_verts()
            if del_type == 'DISSOLVE_E':
                bom.dissolve_edges()
            if del_type == 'DISSOLVE_F':
                bom.dissolve_faces()
            if del_type == 'DISSOLVE_L':
                bom.dissolve_limited()
            if del_type == 'COLLAPSE_E':
                bom.edge_collapse()
            if del_type == 'EDGE_LOOPS':
                bom.delete_edgeloop()
        return {'FINISHED'}


class MESH_OT_restore_merge(Operator):
    bl_idname = "mesh.restore_orig_merge"
    bl_label = "Restore original merge "
    bl_description = "Restores the M shortcut to his original function (merge)"
    bl_options = {'REGISTER', 'UNDO'}

    def execute(self, context):
        restore_orig_keymap()
        return {'FINISHED'}


Scene.merge_delete = EnumProperty(
    items=[
        ("MERGE", "Merge", "Show merge options"),
        ("DELETE", "Delete", "Show delete options"),
    ], name="Merge/Delete auto", description="Show options.")

Scene.merge_type = EnumProperty(
    items=[
        ("CENTER", "At center", "Merge selected at center"),
        ("CURSOR", "At cursor", "Merge selected at cursor"),
        ("COLLAPSE", "Collapse", "Collapse selected points"),
        ("FIRST", "At first", "Merge selected at first"),
        ("LAST", "At last", "Merge selected at last"),
        ("DISTANCE", "By distance", "Merge selected by distance")
    ], name="Merge type", description="Select the mode for merge.")
Scene.merge_distance = FloatProperty(
    name="Merge distance", default=0.0001, precision=4)

Scene.del_type = EnumProperty(
    description="Qué elementos borrar",
    items=[
        ('VERT', "Vertices", ""),
        ('EDGE', "Edges", ""),
        ('FACE', "Faces", ""),
        ('EDGE_FACE', "Only Edges & Faces", ""),
        ('ONLY_FACE', "Only Faces", ""),
        ('DISSOLVE_V', "Dissolve Vertices", ""),
        ('DISSOLVE_E', "Dissolve Edges", ""),
        ('DISSOLVE_F', "Dissolve Faces", ""),
        ('DISSOLVE_L', "Limited Dissolve", ""),
        ('COLLAPSE_E', "Collapse Edges & Faces", ""),
        ('EDGE_LOOPS', "Edge Loops", ""),
    ],
    name="Tipo",
    default='VERT'
)


class VIEW3D_PT_one_click_panel(bpy.types.Panel):
    bl_label = "One click Merge/Delete"
    bl_idname = "VIEW3D_PT_merge_override"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_category = 'Tool'

    @classmethod
    def poll(cls, context):
        return context.mode == 'EDIT_MESH' and context.active_object is not None

    def draw(self, context):
        layout = self.layout
        layout.prop(context.scene, "merge_delete", expand=True)
        col = layout.column(align=True)
        if context.scene.merge_delete == "MERGE":
            col.prop(context.scene, "merge_type", expand=True)
            if context.scene.merge_type == "DISTANCE":
                col = layout.column(align=True)
                col.prop(context.scene, "merge_distance", expand=True)
        else:
            col.prop(context.scene, "del_type", expand=True)

def kmi_bak(kmi, type, idname):
    global orig_kmap_items
    if kmi.type == type and kmi.idname == idname:
        orig_kmap_items.append({'idname': kmi.idname, 'type': kmi.type, 'value': kmi.value,
                                'shift': kmi.shift, 'ctrl': kmi.ctrl, 'alt': kmi.alt,
                                'oskey': kmi.oskey, 'key_modifier': kmi.key_modifier,
                                'active': kmi.active})
        kmi.active = False


def override_keymap():
    global orig_kmap_items

    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        km = kc.keymaps.get('Mesh')
        if not km:
            km = kc.keymaps.new(name='Mesh', space_type='EMPTY')
        for kmi in km.keymap_items:
            kmi_bak(kmi, 'M', 'mesh.merge')
            kmi_bak(kmi, 'DEL', 'mesh.delete')
        kmi = km.keymap_items.new('mesh.merge_auto', type='M', value='PRESS')
        kmi = km.keymap_items.new('mesh.del_auto', type='DEL', value='PRESS')
        return True
    return False


def restore_orig_keymap():
    global orig_kmap_items

    wm = bpy.context.window_manager
    kc = wm.keyconfigs.addon

    if kc:
        km = kc.keymaps.get('Mesh')
        if km:
            for kmi in km.keymap_items:
                if kmi.type == 'M' and kmi.idname == 'mesh.merge': kmi.active = True
                if kmi.type == 'DEL' and kmi.idname == 'mesh.delete': kmi.active = True
                if kmi.idname in ['mesh.merge_auto', 'mesh.del_auto']:
                    km.keymap_items.remove(kmi)
            orig_kmap_items.clear()
            return True
    return False


classes = (MESH_OT_merge_auto, MESH_OT_restore_merge,
           VIEW3D_PT_one_click_panel, MESH_OT_del_auto)

draw_handler = None

def enumPropName(enum, selected):
    scn= bpy.context.scene
    items = scn.bl_rna.properties[enum].enum_items
    #return items[selected].identifier
    return items[selected].name
    #return items[scn.printenum].description


def draw_text_edit_mode():
    context = bpy.context
    obj = context.active_object
    if not obj or obj.type != 'MESH' or obj.mode != 'EDIT':
        return

    font_id = 0
    blf.position(font_id, 40, 10, 0)
    blf.size(font_id, 15)
    mtype=enumPropName('merge_type', context.scene.merge_type)
    dtype=enumPropName('del_type', context.scene.del_type)
    blf.draw(font_id, "M:"+mtype+"|Del:"+dtype) 


def register():
    for cls in classes:
        bpy.utils.register_class(cls)
    override_keymap()
    global draw_handler
    draw_handler = bpy.types.SpaceView3D.draw_handler_add(draw_text_edit_mode, (), 'WINDOW', 'POST_PIXEL')

def unregister():
    restore_orig_keymap()
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)
    global draw_handler
    if draw_handler:
        bpy.types.SpaceView3D.draw_handler_remove(draw_handler, 'WINDOW')
        draw_handler = None













if __name__ == "__main__":
    try:
        unregister()
    except:
        pass

    register()
